/*
 * WebStartEx, launcher for WebStart application that need access to resources outside the jar files.
 * Copyright (C) 2003  Christopher Deckers (chrriis@brainlex.com)
 * http://chrriis.brainlex.com/webstartex
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package chrriis.webstartex;

import java.net.URLDecoder;
import java.util.Enumeration;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.jar.JarOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Set;
import java.util.HashSet;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.jar.JarFile;
import java.net.URL;
import java.lang.ClassLoader;
import java.net.URLClassLoader;
import java.io.ObjectOutputStream;
import java.util.Iterator;
import java.io.FileInputStream;
import java.io.ObjectInputStream;
import java.util.List;
import java.util.Vector;
import java.io.File;
import java.io.FileOutputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedInputStream;
import java.util.StringTokenizer;

/**
 * A proxy class that will load Webstart content.
 * @author Christopher Deckers
 * @version 1.0
 */
public class WebstartEx {

  /**
   * Construct a WebstartEx.
   */
  protected WebstartEx() {}

  /** The property for defining jar index files name with removing option. */
  protected final String INDEX_TO_REMOVE_PROPERTY = "webstartex.remove";
  /** The property for defining jar index files name with keeping option. */
  protected final String INDEX_TO_KEEP_PROPERTY = "webstartex.keep";
  /** The default index name to discover files and remove the content. */
  protected final String DEFAULT_INDEX_TO_REMOVE = "webstartex.remove";
  /** The default index name to discover files and keep the content. */
  protected final String DEFAULT_INDEX_TO_KEEP = "webstartex.keep";
  /** Configuration file name. */
  protected final String CONFIGURATION_FILE = ".webstartex.conf";

  /**
   * Load the configuration.
   * @return The configuration that was loaded, or null if not found.
   */
  protected Configuration loadConfiguration() {
    Configuration configuration = null;
    File configurationFile = new File(System.getProperty("user.dir") + "/" + CONFIGURATION_FILE);
    if(configurationFile.exists()) {
      try {
        ObjectInputStream in = new ObjectInputStream(new BufferedInputStream(new FileInputStream(configurationFile)));
        configuration = (Configuration)in.readObject();
        in.close();
      } catch(Exception e) {}
    }
    return configuration;
  }

  /**
   * Save the configuration file.
   * @param configuration the configuration file to save.
   */
  protected void saveConfiguration(Configuration configuration) {
    File configurationFile = new File(System.getProperty("user.dir") + "/" + CONFIGURATION_FILE);
    try {
      ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream(configurationFile));
      out.writeObject(configuration);
    } catch(Exception e) {}
  }

  /**
   * Load the content of the jar files, using the system property
   * "webstartex.index" to know what files to look for in the different paths.
   * @param applicationJar The name of the application holding the main class,
   * used to generate the cache name.
   */
  protected void loadContent(String applicationJar) {
    String containerArchivePath = getClass().getClassLoader().getResource(getClass().getName().replace('.', '/') + ".class").getPath();
    String downloadPath = null;
    try {
      downloadPath = new File(new URL(containerArchivePath.substring(0, containerArchivePath.lastIndexOf("!/"))).getPath()).getParent();
    } catch(Exception e) {}
    // Look for cache directory
    URL applicationURL = getClass().getClassLoader().getResource(applicationJar);
    if(applicationURL == null)
      throw new IllegalStateException("The jar application \"" + applicationJar + "\" cannot be found!");
    String containerCachePath = applicationURL.getPath();
    String downloadCachePath = null;
    try {
      downloadCachePath = new File(URLDecoder.decode(new URL(containerCachePath.substring(0, containerCachePath.lastIndexOf("!/"))).getPath())).getPath();
    } catch(Exception e) {}
    String userDir = new StringBuffer(downloadCachePath).insert(downloadCachePath.toLowerCase().lastIndexOf(".jar"), "_cache").toString();
    new File(userDir).mkdirs();
    System.setProperty("user.dir", userDir);
    // Do process of copying if needed
    Configuration oldConfiguration = loadConfiguration();
    Configuration newConfiguration = new Configuration();
    if(oldConfiguration == null)
      oldConfiguration = new Configuration();
    String indexToRemove = System.getProperty(INDEX_TO_REMOVE_PROPERTY);
    if(indexToRemove == null)
      indexToRemove = DEFAULT_INDEX_TO_REMOVE;
    String indexToKeep = System.getProperty(INDEX_TO_KEEP_PROPERTY);
    if(indexToKeep == null)
      indexToKeep = DEFAULT_INDEX_TO_KEEP;
    try {
      List list = new Vector();
      Enumeration indexEnumeration = getClass().getClassLoader().getResources(indexToRemove);
      if(indexEnumeration != null)
        while(indexEnumeration.hasMoreElements())
          list.add(indexEnumeration.nextElement());
      indexEnumeration = getClass().getClassLoader().getResources(indexToKeep);
      if(indexEnumeration != null)
        while(indexEnumeration.hasMoreElements())
          list.add(indexEnumeration.nextElement());
//      List list = Collections.list(new URLClassLoader(new URL[] {new File(userDir).toURL()}, getClass().getClassLoader()).getResources(index));
//      for(Iterator i=list.iterator(); i.hasNext(); ) {
//        System.err.println(((URL)i.next()).getPath());
//      }
      // Load file list
      List fileList = oldConfiguration.getExtractedFileList();
      List newFileList = new Vector();
      boolean isConfigurationChanged = false;
      for(Iterator i=list.iterator(); i.hasNext(); ) {
        String filePath = ((URL)i.next()).getPath();
        String archivePath = null;
        try {
          archivePath = new File(URLDecoder.decode(new URL(filePath.substring(0, filePath.lastIndexOf("!/"))).getPath())).getPath();
        } catch(Exception e) {}
        // Check updates
        File archive = new File(archivePath);
        long timeStamp = archive.lastModified();
        long oldTimeStamp = oldConfiguration.getArchiveTimeStamp(archivePath);
        boolean isModified = oldTimeStamp == -1 || timeStamp != oldTimeStamp;
        newConfiguration.setArchiveTimeStamp(archivePath, timeStamp);
        if(isModified) {
          isConfigurationChanged = true;
          JarFile file = new JarFile(archivePath);
          boolean isKeepIndex = false;
          for(Enumeration e=file.entries(); e.hasMoreElements(); ) {
            ZipEntry entry = (ZipEntry)e.nextElement();
            String content = entry.getName();
            boolean isIndex = false;
            if(content.equals(indexToRemove))
              isIndex = true;
            else if(content.equals(indexToKeep)) {
              isIndex = true;
              isKeepIndex = true;
            }
            if(!isIndex) {
              File correspondingFile = new File(userDir + "/" + content);
              correspondingFile.getParentFile().mkdirs();
              if(entry.isDirectory())
                correspondingFile.mkdir();
              else {
                try {
                  InputStream in = new BufferedInputStream(file.getInputStream(entry));
                  OutputStream out = new BufferedOutputStream(new FileOutputStream(correspondingFile));
                  int read;
                  while((read = in.read()) != -1) {
                    out.write(read);
                  }
                  out.flush();
                  out.close();
                  in.close();
                } catch(Exception ex) {
                  ex.printStackTrace();
                }
              }
              newFileList.add(content);
              fileList.remove(content);
            }
          }
          if(!isKeepIndex) {
            // Empty the content of the original archive, if not to keep.
            try {
              JarOutputStream out = new JarOutputStream(new FileOutputStream(archivePath));
              out.putNextEntry(new ZipEntry(indexToRemove));
              if(new File(downloadCachePath).equals(new File(archivePath)))
                out.putNextEntry(new ZipEntry(applicationJar));
//              out.write('.');
              out.flush();
              out.close();
              newConfiguration.setArchiveTimeStamp(archivePath, archive.lastModified());
            } catch(Exception e) {
              e.printStackTrace();
            }
          }
        }
      }
      if(isConfigurationChanged) {
        // Delete files that no longer exist
        for(Iterator i=fileList.iterator(); i.hasNext(); )
          new File((String)i.next()).delete();
        // Save new configuration
        newConfiguration.setExtractedFileList(newFileList);
        saveConfiguration(newConfiguration);
      }
    } catch(Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Starting point of the application.
   * @param args The arguments. The first one must be the name of the jar file
   * to launch. The following are parameters to pass.
   */
  public static void main(String[] args) {
    System.setSecurityManager(null);
    if(args == null || args.length == 0)
      throw new IllegalArgumentException("The first argument must be a jar file.");
    String program = args[0];
    if(!program.toLowerCase().endsWith(".jar"))
      throw new IllegalArgumentException("The first argument must be a jar file.");
    new WebstartEx().loadContent(program);
    Method mainMethod = null;
    try {
      final String codeBase = System.getProperty("user.dir") + "/";
      ClassLoader cl = new URLClassLoader(new URL[] {new File(codeBase + program).toURL()}, ClassLoader.getSystemClassLoader()) {
        protected String findLibrary(String libname) {
          File library = new File(codeBase + System.mapLibraryName(libname));
          if(library.exists())
            return library.getAbsolutePath();
          else
            return super.findLibrary(libname);
        }
      };
      String mainClass = new JarFile(codeBase + program).getManifest().getMainAttributes().getValue("Main-Class");
      mainMethod = Class.forName(mainClass, true, cl).getMethod("main", new Class[] {String[].class});
    } catch(Exception e) {
      e.printStackTrace();
      throw new IllegalStateException("Could not load the jar file.");
    }
    if(mainMethod != null) {
      String[] arguments = new String[args.length - 1];
      for(int i=1; i<args.length; i++)
        arguments[i-1] = args[i];
      try {
        mainMethod.invoke(null, new Object[] {arguments});
      } catch(InvocationTargetException e) {
        e.getTargetException().printStackTrace();
        throw new RuntimeException(e.getMessage());
      } catch(IllegalAccessException e) {
        e.printStackTrace();
        throw new RuntimeException(e.getMessage());
      }
    }
  }

}